/*

######################################################################
######################################################################
##                                                                  ##
##                                                                  ##
##                          netsubs                                 ##
##                                                                  ##
##                neural network subroutines                        ##
##                                                                  ##
##                           by                                     ##
##                                                                  ##
##                    Coryn Bailer-Jones                            ##
##                                                                  ##
##                         22/05/98                                 ##
##                                                                  ##
##             email: calj@mpia-hd.mpg.de                           ##
##               www: http://wol.ra.phy.cam.ac.uk/calj/             ##
##                                                                  ##
##                                                                  ##
##        This file is copyright 1998 by C.A.L. Bailer-Jones        ##
##                                                                  ##
##                                                                  ##
######################################################################
######################################################################


FILE:		netsubs.c
DESCRIPTION:    ANSI C network subroutines
AUTHOR:		Coryn Bailer-Jones
LAST MOD DATE:	09/12/98


######################################################################
######################################################################
##                                                                  ##
##                           netsubs.c                              ##
##                                                                  ##
######################################################################
######################################################################

*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "netsubs.h"

/* error report and exit routine */

void neterr(const char error_text[])
{
  fprintf(stderr,"net ERROR:\n");
  fprintf(stderr," %s\n", error_text);
  fprintf(stderr,"Exiting...\n\n");
  exit(1);
}

/* warning report and return routine */

void netwarn(const char warning_text[])
{
  fprintf(stderr,"net WARNING:\n");
  fprintf(stderr," %s\n\n", warning_text);
  return;
}



/* memory allocations */

int *Ivector(int N)
{
  int *v;

  v = (int *)calloc((unsigned)N, sizeof(int));
  if (!v) neterr("Ivector(): Failed to allocate memory for vector"); 

  return v;
}

double *Dvector(int N)
{
  double *v;

  v = (double *)calloc((unsigned)N, sizeof(double));
  if (!v) neterr("Dvector(): Failed to allocate memory for vector"); 

  return v;
}

/* Creates a vector of pointers to double */
double **Dpvector(int N)
{
  double **v;

  v = (double **)calloc((unsigned)N, sizeof(double *));
  if (!v) neterr("Dpvector(): Failed to allocate memory for vector"); 

  return v;
}

double **Dmatrix(int Nrow, int Ncol)
{
  double **m;
  int i;
  
  m = (double **)calloc((unsigned)Nrow, sizeof(double *));
  if (!m) neterr("Dmatrix(): Failed to allocate memory for 2nd dimension"); 
  for(i=0;i<Nrow;++i) {
    m[i] = (double *)calloc((unsigned)Ncol, sizeof(double));
    if (!m[i]) neterr("Dmatrix(): Failed to allocate memory for 1st dimension");
  }

  return m;
}

double ***Dmatrix3(int Nz, int Nrow, int Ncol)
{
  double ***m;
  int i;
  
  m = (double ***)calloc((unsigned)Nz, sizeof(double *));
  if (!m) neterr("Dmatrix3(): Failed to allocate memory for 3rd dimension"); 
  for(i=0;i<Nz;++i) {
    m[i] = Dmatrix(Nrow, Ncol);
    if (!m[i]) neterr("Dmatrix3(): Can't access Dmatrix()");
  }
  
  return m;
}

/* allocate top dimension of 3-D matrix of pointers */
double ***Dpmatrix3_3(int N)
{
  double ***m;
  
  m = (double ***)calloc((unsigned)N, sizeof(double *));
  if (!m) neterr("Dpmatrix3_3(): Failed to allocate memory for 3rd dimension"); 

  return m;
}

long double *lDvector(int N)
{
  long double *v;

  v = (long double *)calloc((unsigned)N, sizeof(long double));
  if (!v) neterr("lDvector(): Failed to allocate memory for vector"); 

  return v;
}

char *Cvector(int N)
{
  char *c;

  c = (char *)calloc((unsigned)N, sizeof(char));
  if (!c) neterr("Cvector(): Failed to allocate memory for vector"); 

  return c;
}

targets **tmatrix(int Nrow, int Ncol)
{
  targets **m;
  int i;
  
  m = (targets **)calloc((unsigned)Nrow, sizeof(targets *));
  if (!m) neterr("tmatrix(): Failed to allocate memory for 2nd dimension"); 
  for(i=0;i<Nrow;++i) {
    m[i] = (targets *)calloc((unsigned)Ncol, sizeof(targets));
    if (!m[i]) neterr("tmatrix(): Failed to allocate memory for 1st dimension");
  }

  return m;
}

/* allocate top dimension of 3-D matrix of pointers */
targets ***tpmatrix3_3(int N)
{
  targets ***m;
  
  m = (targets ***)calloc((unsigned)N, sizeof(targets *));
  if (!m) neterr("tpmatrix3_3(): Failed to allocate memory for 3rd dimension"); 

  return m;
}


void free_Ivector(int *v, int N)
{
  free(v);
}

void free_Dvector(double *v, int N)
{
  free(v);
}

void free_Dpvector(double **v, int N)
{
  int i;

  for(i=0;i<N;++i)
    free(v[i]);
  free(v);
}

void free_Dmatrix(double **m, int Nrow, int Ncol)
{
  int i;

  for(i=0;i<Nrow;++i)
    free(m[i]);
  free(m);
}

void free_Dmatrix3(double ***m, int Nz, int Nrow, int Ncol)
{
  int i;

  for(i=0;i<Nz;++i)
    free_Dmatrix(m[i], Nrow, Ncol);
  free(m);
}

void free_Dpmatrix3_3(double ***m, int N)
{
  free(m);
}

void free_lDvector(long double *v, int N)
{
  free(v);
}

void free_tmatrix(targets **m, int Nrow, int Ncol)
{
  int i;

  for(i=0;i<Nrow;++i)
    free(m[i]);
  free(m);
}

void free_tpmatrix3_3(targets ***m, int N)
{
  free(m);
}

void free_Cvector(char *c, int N)
{
  free(c);
}


/* general operations */

/* vector times a matrix */
void vectmat(double **w, double *v, int Nrows, int Ncols, double *prod)
{
  /*
  Evaluates prod = v * w = [1*K]*[K*M] ( = (w^T * v^T)^T ).
  w is defined as w[row][column].
  i - row
  j - column
  */

  int i,j;
  
  for(j=0;j<Ncols;++j) {
    prod[j]=0.0;
    for(i=0;i<Nrows;++i) 
      prod[j]+=v[i]*w[i][j];
  }  

}



/* transfer functions */

/* hidden layer function (tanh) */
double Tf(double p, double lambda)
{

  double z;
  double invz;
  double temp;

  temp = (double)(lambda*p);
  if(temp>40.0) { 
    return 1.0;
  }
  else if(temp<-40.0) { 
    return -1.0;
  }
  else {
    z    = exp(temp);
    invz = 1.0/z; 
    return (double)((z-invz)/(z+invz));
  }

}      

/* derivative of hidden layer function, h=f */
double dTf(double f, double lambda)
{
  return lambda*(1.0-f*f);
}

/* output layer function (linear), y=g */
double Tg(double q)
{
  return q;
}      

/* derivative of output layer function */
double dTg(double y)
{
  return 1.0;
}

/* evaluate error */
/* squared error is actually built into function ederiv as it requires
   derivative of error. This is kind of trivial. Hopefully later get a
   better insight into how to implement a more general error. */
double errfn(double y, double t)
{
  return y-t; 
}


/* update the "previous" vector */

void updprev(int N, double *prev, double *cur)
{
  int i;

  for(i=0;i<N;i++)
    prev[i]=cur[i];
}
