# CBJ Oct. 2010
# Plot and create subsets of EID data, including plots for the craterTS paper
# Note that some error bars are so small that plotCI cannot plot them (and gives a warning you can ignore)
# See README for details of data sets

library(gplots)

age.dat <- read.table("eid_age.dat", stringsAsFactors=FALSE)
sym <- c('U', 'L', 'A', 'R', 'S', 'N')
# sym refers to the age uncertainty: A="~" sign was used; N=nothing available; 
# L=lower limit; U=upper limit; R=range given (older age listed); S=second value given (see full table)
diam.dat <- scan("eid_diameter.dat", comment.char="#")
name.dat <- scan("eid_name.dat", comment.char="#", what="character", sep="\n")

########## Plot age distributions of all craters, and generate eid_set1.dat

pdf("EID_data.pdf", width=10, height=7)
par(mfrow=c(2,2), mar=c(3.5,2.0,1.5,1.0), oma=c(1.5,1.5,2.5,2.0), mgp=c(2.5,0.9,0), cex.axis=1.2, cex.lab=1.2)
##### Page 1. Plot all points both over full time range and zooming in
plot(age.dat[,1], rep(1,nrow(age.dat)), type="h", ylim=c(0,1.4), yaxt="n", xlab="Date BP / Myr", ylab="")
plot(age.dat[,1], rep(1,nrow(age.dat)), type="h", ylim=c(0,1.4), yaxt="n", xlab="Date BP / Myr", ylab="", xlim=c(0,250))
# Plot again, but offsetting in y depending on type or estimate (sym), for full time range...
plot(age.dat[,1], rep(1,nrow(age.dat)), type="n", ylim=c(0,6), yaxt="n", yaxs="i", xlab="Date BP / Myr", ylab="")
ind <- which(!is.element(age.dat[,2], sym)) ; points(age.dat[ind,1], rep(1,length.out=length(ind)), pch="|", col="black")
ind <- which( is.element(age.dat[,2], 'L')) ; points(age.dat[ind,1], rep(2,length.out=length(ind)), pch="[", col="red")
ind <- which( is.element(age.dat[,2], 'U')) ; points(age.dat[ind,1], rep(3,length.out=length(ind)), pch="]", col="blue")
ind <- which( is.element(age.dat[,2], 'A')) ; points(age.dat[ind,1], rep(4,length.out=length(ind)), pch="|",  col="green")
ind <- which( is.element(age.dat[,2], 'N')) ; points(age.dat[ind,1], rep(5,length.out=length(ind)), pch="x", col="magenta")
mtext(at=c(1,2,3,4,5), side=2, text=c('E', 'L', 'U', 'A', 'N'))
# ...and zooming in
plot(age.dat[,1], rep(1,nrow(age.dat)), type="n", ylim=c(0,6), yaxt="n", yaxs="i", xlab="Date BP / Myr", ylab="", xlim=c(0,250))
ind <- which(!is.element(age.dat[,2], sym)) ; points(age.dat[ind,1], rep(1,length.out=length(ind)), pch="|", col="black")
ind <- which( is.element(age.dat[,2], 'L')) ; points(age.dat[ind,1], rep(2,length.out=length(ind)), pch="[", col="red")
ind <- which( is.element(age.dat[,2], 'U')) ; points(age.dat[ind,1], rep(3,length.out=length(ind)), pch="]", col="blue")
ind <- which( is.element(age.dat[,2], 'A')) ; points(age.dat[ind,1], rep(4,length.out=length(ind)), pch="|",  col="green")
ind <- which( is.element(age.dat[,2], 'N')) ; points(age.dat[ind,1], rep(5,length.out=length(ind)), pch="x", col="magenta")
mtext(at=c(1,2,3,4,5), side=2, text=c('E', 'L', 'U', 'A', 'N'))
#
mtext("EID. All events", outer=TRUE)
##### Page 2. Plot points which have error bars
unc.ind <- which(!is.element(age.dat[,2], sym))
unc.ind <- unc.ind[order(age.dat[unc.ind,1])] # unc.ind put into order of increasing age
len <- nrow(age.dat[unc.ind,])
yvals <- seq(0,5,0.2)
# Plot points without error bars
plot(age.dat[unc.ind,1], rep(1,len), type="h", ylim=c(0,1.4), yaxt="n", xlab="Date BP / Myr", ylab="")
plot(age.dat[unc.ind,1], rep(1,len), type="h", ylim=c(0,1.4), yaxt="n", xlab="Date BP / Myr", ylab="", xlim=c(0,250))
# Plot offset with error bars, first full age range...
plot(age.dat[,1], rep(1,nrow(age.dat)), type="n", ylim=c(0,5), yaxt="n", xlab="Date BP / Myr", ylab="")
points(age.dat[unc.ind,1], rep(yvals,length.out=len), pch=".", col="black")
plotCI(age.dat[unc.ind,1], rep(yvals,length.out=len), type="n", uiw=as.numeric(age.dat[unc.ind,2]), err='x', gap=0, sfrac=0.01, add=TRUE)
# ...then zooming in
plot(age.dat[,1], rep(1,nrow(age.dat)), type="n", ylim=c(0,5), yaxt="n", xlab="Date BP / Myr", ylab="", xlim=c(0, 250))
points(age.dat[unc.ind,1], rep(yvals,length.out=len), pch="x", col="black")
plotCI(age.dat[unc.ind,1], rep(yvals,length.out=len), type="n", uiw=as.numeric(age.dat[unc.ind,2]), err='x', gap=0, sfrac=0.01, add=TRUE)
#
mtext("EID. All events which have error bars", outer=TRUE)
dev.off()

# Plot just most recent 87 events (set1, but here plotted with errors bars sorted and not using diameter)
yvals <- seq(0,5,0.2)
pdf("eid_set1_altplot.pdf", width=10, height=7)
par(mfrow=c(2,1), mar=c(3.5,2.0,1.5,1.0), oma=c(1.5,1.5,2.5,2.0), mgp=c(2.5,0.9,0), cex.axis=1.2, cex.lab=1.2)
plot(age.dat[unc.ind[1:87],1], rep(1,87), type="h", xlim=c(0,720), ylim=c(0,1.4), yaxt="n", xlab="Date BP / Myr", ylab="")
plot(age.dat[unc.ind[1:87],1], rep(1,87), type="n", xlim=c(0,720), ylim=c(0,5.0), yaxt="n", xlab="Date BP / Myr", ylab="")
points(age.dat[unc.ind[1:87],1], rep(yvals,length.out=87), pch="x", col="black")
plotCI(age.dat[unc.ind[1:87],1], rep(yvals,length.out=87), type="n", uiw=as.numeric(age.dat[unc.ind[1:87],2]), err='x', gap=0, sfrac=0.01, add=TRUE)
mtext("EID. The 87 events which have error bars within the past 700Myr", outer=TRUE)
dev.off()


########## Plot age distributions of craters for certain age, age error or diameter cuts

#
fname <- "eid_set1"
mytit <- "EID set1. The 87 events which have error bars with age<=700Myr"
ind <- which( !is.element(age.dat[,2], sym) & age.dat[,1]<=700)
#
fname <- "eid_set2"
mytit <- "EID set2. The 61 events which have error bars with D>=5km and age<=700Myr"
ind <- which( !is.element(age.dat[,2], sym) & age.dat[,1]<=700 & diam.dat >= 5.0 )
#
fname <- "eid_set3"
mytit <- "EID set3. The 42 events which have error bars with D>=5km and age<=250Myr"
ind <- which( !is.element(age.dat[,2], sym) & age.dat[,1]<=250 & diam.dat >= 5.0 )
#
# NOTE: must also manually supplement the output file with the four data points added to set3ext, as described in the README file
fname <- "eid_set3lim"
mytit <- "EID set3lim. The XX events with D>=5km and age<=250Myr, including 13 events with upper age limits"
ind <- which( !is.element(age.dat[,2], setdiff(sym, 'U')) & age.dat[,1]<=250 & diam.dat >= 5.0 )
#
fname <- "eid_set4"
mytit <- "EID set4. The 32 events which have error bars with D>=5km and age<=150Myr"
ind <- which( !is.element(age.dat[,2], sym) & age.dat[,1]<=150 & diam.dat >= 5.0 )
#
fname <- "eid_set5"
mytit <- "EID set5. The 18 events which have error bars with D>=35km and age<=400Myr"
ind <- which( !is.element(age.dat[,2], sym) & age.dat[,1]<=400 & diam.dat >= 35.0 )
#
fname <- "eid_set6"
mytit <- "EID set6. The 35 events which have error bars with D>=5km, age<=250Myr and sigma(age)<=10Myr"
ind <- which( !is.element(age.dat[,2], sym) & age.dat[,1]<=250 & diam.dat >= 5.0)
ind <- ind[which(as.numeric(age.dat[ind,2])<=10.0)]

# Plot these TS and create output file of ages (sorted into increasing age order)
ind <- ind[order(age.dat[ind,1])] # put ind into order of increasing age
yvals <- seq(0,4,0.2)
pdf(paste(fname,".pdf",sep=""), width=10, height=7)
par(mfrow=c(2,1), mar=c(3.5,3.5,1.5,1.0), oma=c(1.5,1.5,2.5,2.0), mgp=c(2.5,0.9,0), cex.axis=1.2, cex.lab=1.2)
#plot(age.dat[ind,1], rep(1,length(ind)), type="h", xlim=c(0,720), ylim=c(0,1.4), yaxt="n", xlab="Date BP / Myr", ylab="")
plot(age.dat[ind,1], diam.dat[ind], type="h", ylim=c(0, 1.05*max(diam.dat[ind])), yaxs="i", xlab="Date BP / Myr", ylab="Diameter / km")
plot(age.dat[ind,1], rep(1,length(ind)), type="n", ylim=range(yvals), yaxt="n", xlab="Date BP / Myr", ylab="")
points(age.dat[ind,1], rep(yvals,length.out=length(ind)), pch="x", col="black")
plotCI(age.dat[ind,1], rep(yvals,length.out=length(ind)), type="n", uiw=as.numeric(age.dat[ind,2]), err='x', gap=0, sfrac=0.01, add=TRUE)
mtext(mytit, outer=TRUE)
dev.off()
write.table(age.dat[ind,], file=paste(fname,".dat",sep=""), row.names=FALSE, quote=FALSE, col.names=FALSE)

# Plot age-diameter relation
pdf("eid_set3_agediam.pdf", width=10, height=7)
par(mfrow=c(1,1), mar=c(3.5,3.5,1.5,1.0), oma=c(1.5,1.5,2.5,2.0), mgp=c(2.5,0.9,0), cex.axis=1.2, cex.lab=1.2)
plot(age.dat[ind,1], diam.dat[ind], pch=19, ylog=TRUE, xlab="Date BP / Myr", ylab="Diameter / km")
plotCI(age.dat[ind,1], diam.dat[ind], type="n", uiw=as.numeric(age.dat[ind,2]), err='x', gap=0, sfrac=0.01, add=TRUE)
abline(v=seq(from=0,to=2000,by=10), lwd=0.5, col="lightgray")
mytit <- "EID set3. The 42 events which have error bars with D>=5km and age<=250Myr"
mtext(mytit, outer=TRUE)
dev.off()


########## Crate plots and tables for the craterTS paper

# Create latex table of name, age, age error, diameter of eid_set3ext to be used in paper
# Read all in again to ensure that you have NOT done any kind of sorting!
# Still have to hand edit afterwards for final journal format

sym <- c('U', 'L', 'A', 'R', 'S', 'N')
age.dat <- read.table("eid_age.dat", stringsAsFactors=FALSE)
diam.dat <- scan("eid_diameter.dat", comment.char="#")
name.dat <- scan("eid_name.dat", comment.char="#", what="character", sep="\n")

ind <- which( !is.element(age.dat[,2], setdiff(sym, 'U')) & age.dat[,1]<=250 & diam.dat >= 5.0 )
# add in the four data points added to set3ext, as described in the README file
dodo <- c("Avak", "Bosumtwi", "Haughton", "Jebel Waqf as Suwwan")
ind <- union(ind, match(dodo,name.dat))
# put into order of crater name. Actually sort(ind) would do here as name.dat is already is name order
ind <- ind[order(name.dat[ind])]

write.table( cbind(formatC(name.dat[ind], format="s", width=21), "&", formatC(age.dat[ind,1], format="s", width=6, drop0trailing=TRUE), "&", formatC(age.dat[ind,2], format="s", width=5), "&", formatC(diam.dat[ind], format="s", width=5, drop0trailing=TRUE)), file=paste("eid_data.tex",sep=""), row.names=FALSE, quote=FALSE, col.names=FALSE )
# ignore the warnings

#write.table( cbind(formatC(name.dat[ind], format="s", width=21), "&", formatC(age.dat[ind,1], format="f", width=6, digits=2, decimal.mark="&", drop0trailing=TRUE), "&", formatC(age.dat[ind,2], format="s", width=5), "&", formatC(diam.dat[ind], format="s", width=5, drop0trailing=TRUE)), file=paste("eid_data.tex",sep=""), row.names=FALSE, quote=FALSE, col.names=FALSE )


# Plot of set3ext as Diameter vs. age using vertical lines
# starting with ind created for the above table

# Modify the four "ext" data points
age.dat[which(name.dat=='Avak'),1] <- 49.0
age.dat[which(name.dat=='Avak'),2] <- 28.0
age.dat[which(name.dat=='Jebel Waqf as Suwwan'),1] <- 46.5
age.dat[which(name.dat=='Jebel Waqf as Suwwan'),2] <-  5.8
age.dat[which(name.dat=='Bosumtwi'),1] <-  1.07
age.dat[which(name.dat=='Bosumtwi'),2] <-  0.107
age.dat[which(name.dat=='Haughton'),1] <- 39.0
age.dat[which(name.dat=='Haughton'),2] <-  3.9

unc.ind <- ind[which(!is.element(age.dat[ind,2], 'U'))]
ind     <- ind[order(age.dat[ind,1])]         # put into order of increasing age
unc.ind <- unc.ind[order(age.dat[unc.ind,1])] # put into order of increasing age
lim.ind <- setdiff(ind, unc.ind)

# Some points in ind have identical or very close ages which cannot be distinguished in the plots
# So I shift them slightly so one can see this
modage.dat <- age.dat
modage.dat[ 15,1] <- modage.dat[ 15,1] + 0.30
modage.dat[ 75,1] <- modage.dat[ 75,1] + 0.50
modage.dat[ 22,1] <- modage.dat[ 22,1] - 0.10
modage.dat[ 86,1] <- modage.dat[ 86,1] + 0.50
modage.dat[ 24,1] <- modage.dat[ 24,1] - 0.50
modage.dat[  8,1] <- modage.dat[  8,1] - 0.50
modage.dat[ 14,1] <- modage.dat[ 14,1] + 0.30
modage.dat[167,1] <- modage.dat[167,1] + 0.50
modage.dat[125,1] <- modage.dat[125,1] + 0.50

library(gplots)

pdf("craterTSpaper/set3ext_agediam_1.pdf", width=11, height=5)
par(mfrow=c(1,1), mar=c(4.5,5.0,1.0,1.0), oma=c(0.5, 0.5, 0.5, 0.5), mgp=c(2.7,1.0,0), cex.axis=1.5, cex.lab=1.5)
plot(modage.dat[unc.ind,1], diam.dat[unc.ind], type="h", lwd=1, ylim=c(0, 180), yaxs="i", xlab="Time before present / Myr", ylab="Diameter / km")
dev.off()

pdf("craterTSpaper/set3ext_agediam_2_log.pdf", width=11, height=5)
par(mfrow=c(1,1), mar=c(4.5,5.0,1.0,1.0), oma=c(0.5, 0.5, 0.5, 0.5), mgp=c(2.7,1.0,0), cex.axis=1.5, cex.lab=1.5)
plot(age.dat[unc.ind,1], log10(diam.dat[unc.ind]), type="p", pch=19, ylim=c(0.7, 2.3), yaxs="i", xlab="Time before present / Myr", ylab="log10(Diameter / km)")
plotCI(age.dat[unc.ind,1], log10(diam.dat[unc.ind]), type="n", uiw=as.numeric(age.dat[unc.ind,2]), err='x', gap=0, sfrac=0.01, add=TRUE)
points(age.dat[lim.ind,1], log10(diam.dat[lim.ind]), pch="]")
dev.off()

pdf("craterTSpaper/set3ext_agediam_2_logalt.pdf", width=11, height=5)
par(mfrow=c(1,1), mar=c(4.5,5.0,1.0,1.0), oma=c(0.5, 0.5, 0.5, 0.5), mgp=c(2.7,1.0,0), cex.axis=1.5, cex.lab=1.5)
plot(age.dat[unc.ind,1], diam.dat[unc.ind], log="y", type="p", pch=19, ylim=c(4, 200), yaxs="i", xlab="Time before present / Myr", ylab="Diameter / km")
plotCI(age.dat[unc.ind,1], diam.dat[unc.ind], type="n", uiw=as.numeric(age.dat[unc.ind,2]), err='x', gap=0, sfrac=0.01, add=TRUE)
points(age.dat[lim.ind,1], diam.dat[lim.ind], pch="]", col="red")
dev.off()

# Plot all craters which do not have upper/lower age limits or ranges
ind <- which( !is.element(age.dat[,2], c('L', 'U', 'R')))
pdf("craterTSpaper/eid_agediam_1.pdf", width=11, height=5)
par(mfrow=c(1,1), mar=c(4.5,5.0,1.0,1.0), oma=c(0.5, 0.5, 0.5, 0.5), mgp=c(2.7,1.0,0), cex.axis=1.5, cex.lab=1.5)
plot(age.dat[ind,1], diam.dat[ind], log="y", type="h", lwd=1, ylim=c(1, 330), yaxs="i", xlab="Time before present / Myr", ylab="Diameter / km")
dev.off()

# Simple plot showing just the events without diameters or age uncertainties
ind <- which( !is.element(age.dat[,2], sym) & age.dat[,1]<=250 & diam.dat >= 5.0 )
pdf("craterTSpaper/set3_age.pdf", width=7, height=3)
par(mfrow=c(1,1), mar=c(3.1,0.5,0.4,0.5), oma=c(1,1,1,1), mgp=c(2.0,0.7,0), cex.axis=1.0, cex.lab=1.0)
plot(age.dat[ind,1], rep(1,length(ind)), ylim=c(0,1), yaxt="n", yaxs="i", ylab="", xlab="Time / Myr", type="h", lwd=1)
dev.off()
