# C.A.L. Bailer-Jones
# Astrostats 2013 (bug in loglike.sinusoidal() fixed on 2016-05-09)
# This file: sinusoidal.R
# R functions related to the single frequency sinusoidal model

# An event is described by a 4 element vector c(s, s.sd, y, y.sd)
#   where s is the time and y is the signal and s.sd and y.sd their
#   uncertainties, respectively.
# Nevents is the number of events
# obsdata is a matrix of Nevents (rows) and 4 ordered columns c(s, s.sd, y, y.sd)

# Given sinmod and times eventTimes (vector), generate data from a sinusoidal model,
# optionally with additional Gaussian measurement noise of standard deviation ysigma.
# sigma can be a vector of length eventTimes, or a scalar. Default value is zero.
# Return in format obsdata.
gen.sinusoidal <- function(sinmod, eventTimes, ysigma=0) {
  Nevents <- length(eventTimes)
  obsdata <- matrix(data=0, nrow=Nevents, ncol=4)
  obsdata[,1] <- eventTimes
  obsdata[,2] <- 0
  obsdata[,4] <- ysigma
  omegat <- 2*pi*sinmod$freq*obsdata[,1]
  obsdata[,3] <- sinmod$a1*cos(omegat) + sinmod$a2*sin(omegat)
  # exponentially decaying amplitude:
#   obsdata[,3] <- exp(-obsdata[,1]/(diff(range(eventTimes)))) * ( sinmod$a1*cos(omegat) + sinmod$a2*sin(omegat) )
  obsdata[,3] <- obsdata[,3] + rnorm(Nevents, mean=0, sd=obsdata[,4])
  return(obsdata)
}

# Plot obsdata, and optionally overplot a sinusoidal model with parameters sinmod.
# trange=c(tmin,tmax) can be supplied, otherwise (if it's NULL) is calculated from the data.
# If ploterr=TRUE, plot error bars.
# If fname is supplied, save the plot in a PDF with this name.
oplot.obsdata.sinusoidal <- function(obsdata, sinmod=NULL, trange=NULL,
                                     ploterr=TRUE, fname=NULL) {
  library(gplots)
  if(!is.null(fname)) pdf(fname, width=6, height=5)
  par(mfrow=c(1,1), mgp=c(2.0,0.8,0), mar=c(3.5,4.0,1.0,1.0), oma=c(0,0,2.0,0), cex=1.2)
  if(is.null(trange)) {
    trange <- range(c(obsdata[,1]+obsdata[,2], obsdata[,1]-obsdata[,2]))
  }
  if(!is.null(sinmod)) {
    tmod <- seq(from=min(trange), to=max(trange), length.out=1e3)
    omegat <- 2*pi*sinmod$freq*tmod
    ymod <- sinmod$a1*cos(omegat) + sinmod$a2*sin(omegat) 
    ymin <- min(ymod, obsdata[,3]-obsdata[,4])
    ymax <- max(ymod, obsdata[,3]+obsdata[,4])
  } else {
    ymin <- min(obsdata[,3]-obsdata[,4])
    ymax <- max(obsdata[,3]+obsdata[,4])
  }
  plot(obsdata[,1], obsdata[,3], xlim=trange, ylim=c(ymin,ymax),  pch=18,
       xlab="time", ylab="signal")
  if(ploterr) plotCI(obsdata[,1], obsdata[,3], uiw=obsdata[,4], err="y", type="n",
                     gap=0, sfrac=0.01, add=TRUE)
  if(!is.null(sinmod)) {
    lines(tmod, ymod, col="red")
  }
  mtext("measured = black, model = red", padj=-1)
  if(!is.null(fname)) dev.off()
}

# Calculate the Schuster periodogram of obsdata at NFreq uniformly spaced
# frequencies between minFreq and maxFreq.
# Plot this as well as the unnormalized posterior assuming
# (1) a common sigma for all the data equal to the mean of obsdata[,4].
# (2) sigma is unknown
# as well as the other assumptions noted in the script for the
# approximations to hold.
# If fname is supplied, save the plot in a PDF with this name.
schuster <- function(obsdata=obsdata, minFreq=NULL, maxFreq=NULL, NFreq=NULL, fname=NULL) {
  cat("mean, sd of data =", mean(obsdata[,3]), sd(obsdata[,3]), "\n")
  J <- nrow(obsdata)
  sampFreq <- seq(from=minFreq, to=maxFreq, length.out=NFreq)
  schuster <- vector(mode="numeric", length=length(sampFreq))
  for(f in 1:length(sampFreq)) {
    omegat <- 2*pi*sampFreq[f]*obsdata[,1]
    R <- sum(obsdata[,3]*cos(omegat))
    I <- sum(obsdata[,3]*sin(omegat))
    schuster[f] <- (R^2 + I^2)/J
  }
  schusterNorm <- schuster/( sum(schuster)*(maxFreq-minFreq)/NFreq ) # set area to unity
  #
  if(!is.null(fname)) pdf(fname, width=7, height=6)
  par(mfrow=c(3,1), mar=c(0.5,3.5,0,0), oma=c(3.0,0,1,1), mgp=c(2.0,0.6,0), cex=1.1)
  plot(sampFreq, schusterNorm, xaxt="n", ylab="power (normalized)", type="l")
  text(max(sampFreq), 0.9*max(schusterNorm), "Schuster periodogram", pos=2)
  # posterior periodogram with known common sigma
  logpost <- (1/log(10))*schuster/mean(obsdata[,4])^2
  plot(sampFreq, logpost, xaxt="n", ylab="log10post", type="l")
  text(max(sampFreq), 0.9*max(logpost), "assuming common sigma", pos=2)
  # posterior periodogram with unknown sigma
  # print(sum(obsdata[,3]^2)/2 - schuster) # check: are always positive
  posterior <- (sum(obsdata[,3]^2)/2 - schuster)^((2-J)/2)
  plot(sampFreq, log10(posterior), ylab="log10post", type="l")
  text(max(sampFreq), 0.7*max(log10(posterior)), "assuming unknown sigma", pos=2)
  #
  mtext("frequency", side=1, line=1.5, outer=TRUE, cex=1.1)
  if(!is.null(fname)) dev.off()
}

# Return log10(unnormalized posterior) of the sinusoidal model
# (see notes on the functions called)
logpost.sinusoidal <- function(theta, obsdata, alpha, ind=NULL) {
  logprior <- logprior.sinusoidal(theta, alpha)
  if(is.finite(logprior)) { # only evaluate model if parameters are sensible
    return( loglike.sinusoidal(theta, obsdata, ind) + logprior )
  } else {
    return(-Inf)
  }
}

# Return log10(likelihood) of the sinusoidal model on rows ind of 
# obsdata (to within an additive constant)
# with parameters theta = c(a1, a2, freq)
# ... is needed to pick up the unwanted alpha passed by kfoldcv(),
loglike.sinusoidal <- function(theta, obsdata, ind=NULL, ...) {
  if(is.null(ind)) ind <- 1:nrow(obsdata)
  omegat <- 2*pi*theta[3]*obsdata[ind,1]
  modpred <- theta[1]*cos(omegat) + theta[2]*sin(omegat) 
  logEventLike <- (1/log(10))*dnorm(x=obsdata[ind,3], mean=modpred,
                                    sd=obsdata[ind,4], log=TRUE)
  return( sum(logEventLike) )
}

# Return log10(unnormalized prior) of the sinusoidal model
# with parameters theta = c(a1, a2, freq) and selected prior
# hyperparameters alpha
logprior.sinusoidal <- function(theta, alpha) {
  a1Prior <- dnorm(x=theta[1], mean=0, sd=alpha[1])
  a2Prior <- dnorm(x=theta[2], mean=0, sd=alpha[2])
  freqPrior <- dgamma(x=theta[3], shape=1.5, scale=alpha[3])
  return( sum(log10(a1Prior), log10(a2Prior), log10(freqPrior)) )
}

# return Nsamp samples from prior
# (is consistent with logprior.sinusoidal)
sampleprior.sinusoidal <- function(Nsamp, alpha) {
  a1 <- rnorm(n=Nsamp, mean=0, sd=alpha[1])
  a2 <- rnorm(n=Nsamp, mean=0, sd=alpha[2])
  freq <- rgamma(n=Nsamp, shape=1.5, scale=alpha[3])
  return(cbind(a1, a2, freq))
}
