# C.A.L. Bailer-Jones
# Astrostats 2013
# This file: quadraticmodel.R
# Functions to provide evaluations of prior, likelihood and posterior for quadratic model 
# plus sampling from the prior

# theta is vector of parameters; obsdata is 2 column dataframe [x,y].
# the priors are hard-wired into the functions

# return log10(unnormalized posterior) of the quadratic model
# (see notes on the functions called)
logpost.quadraticmodel <- function(theta, obsdata, ind=NULL) {
  logprior <- logprior.quadraticmodel(theta)
  if(is.finite(logprior)) { # only evaluate model if parameters are sensible
    return( loglike.quadraticmodel(theta, obsdata, ind) + logprior )
  } else {
    return(-Inf)
  }
}

# return log10(likelihood) for parameters theta for rows ind in obsdata
# (do all if ind not specified)
# dnorm(..., log=TRUE) returns log base e, so multiply by 1/ln(10) = 0.4342945
# to get log base 10
loglike.quadraticmodel <- function(theta, obsdata, ind=NULL) {
  if(is.null(ind)) ind <- 1:nrow(obsdata)
  # convert alpha to a_1 and log10(ysig) to ysig
  theta[2] <- tan(theta[2])
  theta[4] <- 10^theta[4]
  modPred <- drop( theta[1:3] %*% t(cbind(1,obsdata[ind,]$x,obsdata[ind,]$x^2)) )
  # Dimensions in mixed vector matrix multiplication: [Ndat] = [P] %*% [P x Ndat] 
  logLike <- (1/log(10))*sum( dnorm(modPred - obsdata[ind,]$y, mean=0, sd=theta[4], log=TRUE) )
  return(logLike)
}

# return log10(unnormalized prior)
logprior.quadraticmodel <- function(theta) {
  a0Prior      <- dnorm(theta[1], mean=0, sd=4)
  alphaPrior   <- 1
  a2Prior      <- dnorm(theta[3], mean=0, sd=1)
  logysigPrior <- 1 
  logPrior <- sum( log10(a0Prior), log10(alphaPrior), log10(a2Prior), log10(logysigPrior) )
  return(logPrior)
}

# return Nsamp samples from prior
# NOTE: This uses a proper gamma prior on ysig instead of an improper Jeffreys prior,
# which is inconsistent with logprior.quadraticmodel(). This is generally a BAD IDEA
# and really we should modify logprior.quadraticmodel() to use proper priors.
sampleprior.quadraticmodel <- function(Nsamp) {
  a0 <- rnorm(Nsamp, mean=0, sd=4)
  a1 <- tan(runif(Nsamp, min=-pi/2, max=pi/2))
  a2 <- rnorm(Nsamp, mean=0, sd=1)
  logysig <- rgamma(Nsamp, shape=1.5, scale=1)
  return(cbind(a0, a1, a2, logysig))
}
