# C.A.L. Bailer-Jones
# Astrostats 2013
# This file: monte_carlo.R
# R code for Metropolis algorithm (and for constructing covariance matrix)

library(mvtnorm) # for rmvnorm()

# Metropolis (MCMC) algorithm to sample from function func()
# The first argument of func must be a real vector of parameters, the initial values
# of which are provided by the real vector thetaInit.
# func() returns a two-element vector, the logPrior and logLike (log base 10), the sum
# of which is taken to be the log of the density function (i.e. unnormalized posterior).
# The MCMC sampling PDF is the multivariate Gaussian with fixed covariance, sampleCov.
# A total of Nburnin+Nsamp samples are drawn, of which the last Nsamp are kept.
# As the sampling PDF is symmetric, the Hasting factor cancels,
# leaving the basic Metropolis algorithm.
# Diagnostics are printed very verbose^th sample: sample number, acceptance rate so far.
# ... is used to pass data, prior parameters etc. to func()
# Return a Nsamp * (2+Ntheta) matrix (no names), where the columns are
# 1:  log10 prior PDF
# 2:  log10 likelihood
# 3+: Ntheta parameters
# (The order of the parameters in thetaInit and sampleCov must match, of course.)
metrop <- function(func, thetaInit, Nburnin, Nsamp, verbose, sampleCov, ...) {

  Ntheta   <- length(thetaInit)
  thetaCur <- thetaInit
  funcCur  <- func(thetaInit, ...) # log10
  funcSamp <- matrix(data=NA, nrow=Nsamp, ncol=2+Ntheta) # this will be filled and returned
  nAccept  <- 0
  acceptRate <- 0
  
  for(n in 1:(Nburnin+Nsamp)) {

    # Metropolis algorithm. No Hastings factor for symmetric sampling distributions.
    thetaProp <- rmvnorm(n=1, mean=thetaCur, sigma=sampleCov, method="eigen")
    funcProp  <- func(thetaProp, ...) 
    logMR <- sum(funcProp) - sum(funcCur) # log10 of the Metropolis ratio 
    if(logMR>=0 || logMR>log10(runif(1, min=0, max=1))) {
      thetaCur   <- thetaProp
      funcCur    <- funcProp
      nAccept    <- nAccept + 1
      acceptRate <- nAccept/n
    }
    if(n>Nburnin) {
      funcSamp[n-Nburnin,1:2] <- funcCur
      funcSamp[n-Nburnin,3:(2+Ntheta)] <- thetaCur
    }

    # diagnostics
    if( is.finite(verbose) && (n%%verbose==0 || n==Nburnin+Nsamp) ) {
      sdump1 <- noquote( formatC(n,          format="d", digits=5, flag="") )
      sdump2 <- noquote( formatC(Nburnin,    format="g", digits=5, flag="") )
      sdump3 <- noquote( formatC(Nsamp,      format="g", digits=5, flag="") )
      sdump4 <- noquote( formatC(acceptRate, format="f", width=7, digits=4, flag="") )
      cat(sdump1, "of", sdump2, "+", sdump3, sdump4, "\n")
    }

  }

  return(funcSamp)
 
}

# Return a covariance matrix given a vector of the standard deviations
# and the global (i.e. scalar) correlation coefficient.
# This is inefficient, but matrix is very small and only done once.
make.covariance.matrix <- function(sampleSD, sampleCor) {
  Len <- length(sampleSD)
  covMatrix <- matrix(nrow=Len, ncol=Len)
  if(abs(sampleCor)>1) {stop("|sampleCor| > 1")}
  for(i in 1:(Len)) {
    for(j in 1:(Len)) {
      if(i==j) {
        covMatrix[i,j] <- sampleSD[i]^2
      } else {
        covMatrix[i,j] <- sampleCor*sampleSD[i]*sampleSD[j]
      }
    }
  }
  return(covMatrix)
}
