# C.A.L. Bailer-Jones
# Astrostats 2013
# This file: exp_sinusoidal.R
# R code for experimenting with the sinusoidal model

source("sinusoidal.R")
source("monte_carlo.R") # provides metrop() and make.covariance.matrix()

set.seed(200)

# Generate a time series from a sinusoidal model
sinmod <- list(a1=1, a2=1, freq=0.5)
eventTimes <- sort(c(0, 20, runif(n=23, min=0, max=20)))
ysigma <- 0.25
obsdata <- gen.sinusoidal(sinmod=sinmod, eventTimes=eventTimes, ysigma=ysigma)

# Plot data and overplot with true sinusoidal model
oplot.obsdata.sinusoidal(obsdata=obsdata, sinmod=sinmod, ploterr=TRUE, fname="sindat1.pdf")
oplot.obsdata.sinusoidal(obsdata=obsdata, sinmod=NULL,   ploterr=TRUE, fname="sindat2.pdf")
oplot.obsdata.sinusoidal(obsdata=obsdata, sinmod=sinmod, ploterr=FALSE)

# Model assumes data are zero mean
obsdata[,3] <- obsdata[,3] - mean(obsdata[,3])

# Calculate and plot Schuster periodogram and derived posterior PDFs over frequency
schuster(obsdata=obsdata, minFreq=0.05, maxFreq=1, NFreq=1e3, fname="schuster.pdf")

# For the following inference we sample over the three parameters
# theta = c(a1, a2, freq).  The hyperparameters of the prior PDFs are
# alpha = c(sd_a1, sd_a2, scale_freq), with mean_a1, mean_a2, and
# shape_freq fixed in the definitions of the functions.
alpha <- c(1, 1, 1) # c(sd_a1, sd_a2, scale_freq)

# Use MCMC to sample posterior for sinusoidal model: c(a1, a2, freq)
sampleCov <- make.covariance.matrix(sampleSD=c(0.05, 0.05, 0.1), sampleCor=0)
thetaInit <- c(0.5, 2, 0.3)
postSamp <- metrop(func=logpost.sinusoidal, thetaInit=thetaInit, Nburnin=0, 
                   Nsamp=3e4, verbose=1e3, sampleCov=sampleCov, 
                   obsdata=obsdata, alpha=alpha)

# Plot MCMC chains and use density estimation to plot 1D posterior PDFs from these.
thetaTrue <- as.numeric(sinmod)
parnames <- c("a1", "a2", "freq")
pdf("sindat_mcmc.pdf", width=7, height=6)
par(mfrow=c(3,2), mar=c(3.0,3.0,0.5,0.5), oma=c(1,1,1,1), mgp=c(1.8,0.6,0), cex=1.0)
for(p in 3:5) { # columns of postSamp
  plot(1:nrow(postSamp), postSamp[,p], type="l", xlab="iteration", ylab=parnames[p-2])
  postDen <- density(postSamp[,p], n=2^10)
  plot(postDen$x, postDen$y, type="l", xlab=parnames[p-2], ylab="density")
  abline(v=thetaTrue[p-2], col="red")
}
dev.off()

# Find MAP solution and mean solutions and overplot on data
posMAP     <- which.max(postSamp[,1]+postSamp[,2]) 
(thetaMAP  <- postSamp[posMAP, 3:5])
(thetaMean <- apply(postSamp[,3:5], 2, mean)) # Monte Carlo integration
locplot <- function(theta, col) {
  tsamp <- seq(from=min(eventTimes), to=max(eventTimes), length.out=1e3)
  omegat <- 2*pi*theta[3]*tsamp
  modpred <- theta[1]*cos(omegat) + theta[2]*sin(omegat) 
  lines(tsamp, modpred, col=col, lw=1) 
}
pdf("sindat_fits.pdf", width=6, height=5)
par(mfrow=c(1,1), mgp=c(2.0,0.8,0), mar=c(3.5,4.0,1.0,1.0), oma=c(0,0,0,0), cex=1.2)
plotCI(obsdata[,1], obsdata[,3], xlab="time", ylab="signal", uiw=obsdata[,4], gap=0)
locplot(theta=as.numeric(sinmod), col="red")
locplot(theta=thetaMAP,  col="blue")
locplot(theta=thetaMean, col="green")
dev.off()
