# C.A.L. Bailer-Jones
# Astrostats 2013
# This file: exp_OUprocess.R
# R code for experimenting with modelling the OU process

source("OUprocess.R")
source("monte_carlo.R")

set.seed(102)

# Generate a time series from an OU process
ouproc <- list(diffcon=1, relax=1, zstartmean=0, zstartsd=0)
eventTimes <- sort(c(0, 20, runif(n=48, min=0, max=20)))
ysigma <- 0.1
temp <- gen.OUprocess(ouproc=ouproc, eventTimes=eventTimes, ysigma=ysigma)
obsdata  <- temp$obsdata
procDist <- temp$procDist

# Plot data, ... first overplotting the distributions for each point from the true process
# i.e. the mean and sd of the update equation
oplot.obsdata.OUprocess(obsdata=obsdata, procDist=procDist, fname="ouprocdat1.pdf")
# ... then overplotting the prior PDF evaluated for the OU process at the given parameters
oplot.obsdata.OUprocess(obsdata=obsdata, ouproc=ouproc, fname="ouprocdat2.pdf")

# Model assumes data are zero mean
obsdata[,3] <- obsdata[,3] - mean(obsdata[,3])

# For the following inference, we fix the OU process parameters
# ouprocFixed <- c(zstartmean, zstartsd) and just sample over the
# parameters theta = c(diffcon, relax). The scale hyperparameters on the
# gamma priors are specified by alpha = c(scale_diffcon, scale_relax)
ouprocFixed <- c(0,0) # c(zstartmean, zstartsd)
alpha <- c(1,1) # c(scale_diffcon, scale_relax)
# Or, using the prescription in the lecture
# trange <- diff(range(obsdata[,1]))/10
# alpha <- c(2*var(obsdata[,3])/trange, trange)

# Use MCMC to sample posterior for OU process: c(diffcon, relax)
sampleCov <- make.covariance.matrix(sampleSD=c(0.1, 0.1), sampleCor=0)
thetaInit <- c(3,3)
set.seed(150)
postSamp <- metrop(func=logpost.OUprocess, thetaInit=thetaInit, Nburnin=0, 
                   Nsamp=1e4, verbose=1e3, sampleCov=sampleCov, 
                   obsdata=obsdata, ouprocFixed=ouprocFixed, alpha=alpha)

# Plot MCMC chains and use density estimation to plot 1D posterior PDFs from these.
thetaTrue <- as.numeric(ouproc[1:2])
parnames <- c("diffcon", "relax")
pdf("ouprocdat_mcmc.pdf", width=7, height=6)
par(mfrow=c(2,2), mar=c(3.0,3.0,0.5,0.5), oma=c(1,1,1,1), mgp=c(1.8,0.6,0), cex=1.0)
for(p in 3:4) { # columns of postSamp
  plot(1:nrow(postSamp), postSamp[,p], type="l", xlab="iteration", ylab=parnames[p-2])
  postDen <- density(postSamp[,p], n=2^10)
  plot(postDen$x, postDen$y, type="l", xlab=parnames[p-2], ylab="density")
  abline(v=thetaTrue[p-2], col="red")
}
dev.off()
