# C.A.L. Bailer-Jones
# Astrostats 2013
# This file: OUprocess.R
# R functions related to the OU process

# An event is described by a 4 element vector c(s, s.sd, y, y.sd)
#   where s is the time and y is the signal and s.sd and y.sd their
#   uncertainties, respectively.
# Nevents is the number of events
# obsdata is a matrix of Nevents (rows) and 4 ordered columns c(s, s.sd, y, y.sd)
# ouproc is a named list of the parameters of the OU process
#   ouproc = list(diffcon, relax, zstartmean, zstartsd)
#   zstartmean and zstartsd are the initial conditions.

# Given ouproc and times eventTimes (vector), generate data from an OU process
# using the update equations.
# If ysigma>0 add Gaussian measurement noise with this standard deviation.
#   ysigma can be a vector of length eventTimes, or a scalar. Default value is zero.
# Return a two element list:
# obsdata
# procDist, a dataframe with Nevents rows and 2 columns:
#   mean and variance of the process itself at each time step
gen.OUprocess <- function(ouproc, eventTimes, ysigma=0) {
  Nevents <- length(eventTimes)
  obsdata <- matrix(data=0, nrow=Nevents, ncol=4)
  zMean    <- vector(mode="numeric", length=Nevents)
  zVar     <- vector(mode="numeric", length=Nevents)
  obsdata[,1] <- eventTimes
  obsdata[,2] <- 0
  zMean[1] <- ouproc$zstartmean
  zVar[1]  <- ouproc$zstartsd
  obsdata[1,3] <- ouproc$zstartmean + rnorm(1, mean=0, sd=ouproc$zstartsd)
  for(j in 2:Nevents) {
    nu <- exp(-(obsdata[j,1]-obsdata[j-1,1])/ouproc$relax)
    Vz <- (ouproc$diffcon*ouproc$relax/2)*(1-nu^2)
    zMean[j] <- obsdata[j-1,3]*nu
    zVar[j]  <- Vz
    obsdata[j,3] <- rnorm(1, mean=zMean[j], sd=sqrt(zVar[j]))
  }
  if(ysigma>0) { # add measurement noise
    obsdata[,4] <- ysigma
    obsdata[,3] <- obsdata[,3] + rnorm(Nevents, mean=0, sd=obsdata[j,4])
  }
  return(list(obsdata=obsdata, procDist=data.frame(zMean=zMean, zVar=zVar)))
}

# Plot obsdata, and optionally overplot either (if defined)
# (1) points with mean and variance given by procDist, or
# (2) predicted priors of an OU process evaluated on these data with parameters ouproc.
# If both are defined then only (1) is plotted. (This will usually be used to 
# plot the mean and sd of the update formula used to generate obsdata.)
# The data are plotted in black, procDist in magenta, the predicted priors in red.
# trange=c(tmin,tmax) can be supplied, otherwise (if it's NULL) is calculated from the data.
# If ploterr=TRUE, plot error bars on the data.
# If fname is supplied, save the plot in a PDF with this name.
# NOTE: This will give warnings if the error bars are too small to plot
oplot.obsdata.OUprocess <- function(obsdata, procDist=NULL, ouproc=NULL, trange=NULL,
                                    ploterr=TRUE, fname=NULL) {
  library(gplots)
  if(!is.null(fname)) pdf(fname, width=6, height=5)
  if(is.null(trange)) {
    trange <- range(c(obsdata[,1]+obsdata[,2], obsdata[,1]-obsdata[,2]))
  }
  if(!is.null(ouproc)) {
    procParam <- eval.OUprocess(ouproc, obsdata)
  }
  if(!is.null(procDist)) {
    procParam <- procDist
  }
  par(mfrow=c(1,1), mgp=c(2.0,0.8,0), mar=c(3.5,3.5,1.0,1.0), oma=c(0,0,1,0), cex=1.1)
  if(!is.null(procParam)) {
    ymin <- min(c(procParam$zMean-sqrt(procParam$zVar), obsdata[,3]-obsdata[,4]))
    ymax <- max(c(procParam$zMean+sqrt(procParam$zVar), obsdata[,3]+obsdata[,4]))
  } else {
    ymin <- min(obsdata[,3]-obsdata[,4])
    ymax <- max(obsdata[,3]+obsdata[,4])
  }
  plot(obsdata[,1], obsdata[,3], xlim=trange, ylim=c(ymin,ymax),  pch=18,
       xlab="time", ylab="signal")
  if(ploterr) plotCI(obsdata[,1], obsdata[,3], uiw=obsdata[,4], err="y", type="n",
                     gap=0, sfrac=0.01, add=TRUE)
  if(!is.null(procDist)) {
    points(obsdata[,1], procDist$zMean, pch=8, col="magenta")
    plotCI(obsdata[,1], procDist$zMean, uiw=sqrt(procDist$zVar), err="y", type="n",
           col="magenta", gap=0, sfrac=0.01, add=TRUE)
  }
  if(!is.null(ouproc) && is.null(procDist)) {
    points(obsdata[,1], procParam$zMean, pch=8, col="red")
    plotCI(obsdata[,1], procParam$zMean, uiw=sqrt(procParam$zVar), err="y", type="n",
           col="red", gap=0, sfrac=0.01, add=TRUE)
  }
  mtext("measured = black, true process = magenta, model prior = red", padj=-1)
  if(!is.null(fname)) dev.off()
}

# Evaluate an OU process of given parameters using given obsdata.
# Specifially, return a ncol(obsdata) x 2 dataframe with named columns
# zMean and zVar which give the mean and variance of the prior PDF
# of the OU process variable at each event.
# Note: probably slower than it should be due to presence of loop
eval.OUprocess <- function(ouproc, obsdata) {
  Nevents  <- nrow(obsdata)
  zMean    <- vector(mode="numeric", length=Nevents)
  zVar     <- vector(mode="numeric", length=Nevents)
  j <- 1
  zMean[j] <- ouproc$zstartmean
  zVar[j]  <- ouproc$zstartsd^2
  for(j in 1:Nevents) {
    if(j > 1) {
      # warning: divide by zero if relax=0 (controlled by prior)
      nu <- exp(-(obsdata[j,1]-obsdata[j-1,1])/ouproc$relax)
      Vz <- (ouproc$diffcon*ouproc$relax/2)*(1-nu^2)
      zMean[j] <- zMeanPost*nu # + ouproc$offset*(1-nu)
      zVar[j]  <- zVarPost*nu^2 + Vz
    }
    if(j < Nevents) { # calculate posterior
      ysigsq    <- obsdata[j,4]^2
      denom     <- ysigsq + zVar[j]
      if(denom==0) { # controlled by data/prior
        stop("Both ysigsq and zVar are zero, so posterior moments are undefined")
      }
      zMeanPost <- (obsdata[j,3]*zVar[j] + zMean[j]*ysigsq)/denom
      zVarPost  <- zVar[j]*ysigsq/denom
    }
  }
  return(data.frame(zMean=zMean, zVar=zVar))
}  

# Return log10(unnormalized posterior) of the OU process
# (see notes on functions called)
logpost.OUprocess <- function(theta, obsdata, ouprocFixed, alpha, ind=NULL) {
  logprior <- logprior.OUprocess(theta, alpha)
  if(is.finite(logprior)) { # only evaluate model if parameters are sensible
    return( loglike.OUprocess(theta, obsdata, ouprocFixed, ind) + logprior )
  } else {
    return(-Inf)
  }
}

# Return log10(likelihood) of the OU process defined by parameters
# theta = c(diffcon, relax) and ouprocFixed = c(zstartmean, zstartsd)
# for events ind in the time series obsdata (to within an additive
# constant). We require the full set of data in order to propagate the
# estimates of the process parameters along the chain of
# events. However, the likelihood is calculated and returned only for
# the events specified in ind. (This can be used for calculating the
# k-fold CV likelihood, for example.) If ind=NULL (the default), the
# likelihood for all the data is calculated.
# ... is needed to pick up the unwanted alpha passed by kfoldcv().
# Note: Function is much slower than other loglike functions, presumably
# due to eval.OUprocess()
loglike.OUprocess <- function(theta, obsdata, ouprocFixed, ind=NULL, ...) {
  if(is.null(ind)) ind <- 1:nrow(obsdata)
  ouproc <- list(diffcon=theta[1], relax=theta[2], zstartmean=ouprocFixed[1],
                 zstartsd=ouprocFixed[2])
  procParam <- eval.OUprocess(ouproc, obsdata)
  yMean <- procParam$zMean
  yVar  <- obsdata[,4]^2 + procParam$zVar
  logEventLike <- (1/log(10))*dnorm(x=obsdata[ind,3], mean=yMean[ind],
                                    sd=sqrt(yVar[ind]), log=TRUE)
  return( sum(logEventLike) )
}

# Return log10(unnormalized prior) unnormalized prior of the OU process
# with parameters theta = c(diffcon, relax) and selected prior
# hyperparameters alpha.
# zstartmean and zstartsd are assumed fixed.
logprior.OUprocess <- function(theta, alpha) {
  diffconPrior <- dgamma(x=theta[1], shape=1.5, scale=alpha[1])
  relaxPrior   <- dgamma(x=theta[2], shape=1.5, scale=alpha[2])
  return( sum(log10(diffconPrior), log10(relaxPrior)) )
}

# return Nsamp samples from prior
# (is consistent with logprior.OUprocess)
sampleprior.OUprocess <- function(Nsamp, alpha) {
  diffcon <- rgamma(n=Nsamp, shape=1.5, scale=alpha[1])
  relax   <- rgamma(n=Nsamp, shape=1.5, scale=alpha[2])
  return(cbind(diffcon, relax))
}
